#include <msp430.h>
#include <inttypes.h>

#define UART_TX_POUT		P1OUT
#define UART_TX_PDIR		P1DIR
#define UART_TX_PSEL		P1SEL
#define UART_TX			0x02

extern volatile unsigned int RXTXData;
extern volatile unsigned char BitCnt;


void uart_init(void) {
	UART_TX_POUT |= UART_TX;
	UART_TX_PDIR |= UART_TX;
	UART_TX_PSEL |= UART_TX;
  	BCSCTL1 |= DIVA_3;                        // ACLK = LFXT1CLK/8
  	CCTL0 = OUT;                              // TXD Idle as Mark
  	TACTL = TASSEL_2 + MC_2;                  // SMCLK, continuous mode
	__nop();
	__bis_SR_register(GIE);
	__nop();
}

void uart_tx_byte(uint8_t tx) {
	RXTXData = tx;
  	BitCnt = 0xA;                             // Load Bit counter, 8data + ST/SP
  	CCR0 = TAR;                               // Current state of TA counter
  	CCR0 += 1000;                           // Some time till first bit
  	RXTXData |= 0x100;                        // Add mark stop bit to RXTXData
  	RXTXData = RXTXData << 1;                 // Add space start bit
  	CCTL0 = OUTMOD0 + CCIE;                   // TXD = mark = idle
  	while ( CCTL0 & CCIE );                   // Wait for TX completion
}

/* 
 * gps_transmit_string
 *
 * transmits a command to the GPS
 */
void gps_transmit_string(const unsigned char *cmd, uint8_t length) {
	uint8_t i;

	for (i = 0; i < length; i++) {
		uart_tx_byte(cmd[i]);
	}
}

/* 
 * gps_disable_nmea_output
 *
 * disables all NMEA messages to be output from the GPS.
 * even though the parser can cope with NMEA messages and ignores them, it 
 * may save power to disable them completely.
 *
 * returns if ACKed by GPS
 *
 */
const unsigned char nonmea[] = {
	0xB5, 0x62, 0x06, 0x00, 20, 0x00,		/* UBX-CFG-PRT */
	0x01, 0x00, 0x00, 0x00, 			/* UART1, reserved, no TX ready */
	0xe0, 0x08, 0x00, 0x00,				/* UART mode (8N1) */
	0x80, 0x25, 0x00, 0x00,				/* UART baud rate (9600) */
	0x01, 0x00,					/* input protocols (uBx only) */
	0x01, 0x00,					/* output protocols (uBx only) */
	0x00, 0x00,					/* flags */
	0x00, 0x00,					/* reserved */
	0xaa, 0x79					/* checksum */
};
void gps_disable_nmea_output(void) {

	gps_transmit_string(nonmea, sizeof(nonmea));
	return;
}


#define TP_PERIOD	1000000UL
#define TP_LEN		0UL
#define TP_PERIOD_LOCKED	100UL
#define TP_LEN_LOCKED		50UL
	
const unsigned char timepulse[40] = {
	0xB5, 0x62, 0x06, 0x31, 32, 0x00,	/* UBX-CFG-TP5 */
	0x00, 0x01, 0x00, 0x00,			/* timepulse1, version 1, reserved1 */
	0x00, 0x00, 0x00, 0x00,			/* no cable delay, no RF group delay */
	(uint8_t) (TP_PERIOD),			/* period time, little endian */
	(uint8_t) (TP_PERIOD >> 8),
	(uint8_t) (TP_PERIOD >> 16),
	(uint8_t) (TP_PERIOD >> 24),
	(uint8_t) (TP_PERIOD_LOCKED),			/* period time, little endian */
	(uint8_t) (TP_PERIOD_LOCKED >> 8),
	(uint8_t) (TP_PERIOD_LOCKED >> 16),
	(uint8_t) (TP_PERIOD_LOCKED >> 24),
	(uint8_t) (TP_LEN),			/* pulse length, little endian */
	(uint8_t) (TP_LEN >> 8),
	(uint8_t) (TP_LEN >> 16),
	(uint8_t) (TP_LEN >> 24),
	(uint8_t) (TP_LEN_LOCKED),			/* pulse length, little endian */
	(uint8_t) (TP_LEN_LOCKED >> 8),
	(uint8_t) (TP_LEN_LOCKED >> 16),
	(uint8_t) (TP_LEN_LOCKED >> 24),
	0x00, 0x00, 0x00, 0x00,			/* no user delay */
	0b00010111, 0x00, 0x00, 0x00,		/* enable TP, lock to GNSS, setting is length */
	0x96, 0xa6				/* placeholder for checksum */
};

void gps_enable_timepulse(void) {
	int i;
	/*
	for (i = 2; i < sizeof(timepulse)-2; i++) {
		timepulse[38] += timepulse[i];		
		timepulse[39] += timepulse[38];
	}
	*/

	gps_transmit_string(timepulse, sizeof(timepulse));

	return;
}

/*
 * gps_set_gps_only
 *
 * tells the uBlox to only use the GPS satellites
 *
 * returns if ACKed by GPS
 *
 */
const unsigned char gpsonly[] = {
	0xB5, 0x62, 0x06, 0x3E, 36, 0x00,		/* UBX-CFG-GNSS */
	0x00, 32, 32, 4,				/* use 32 channels, 4 configs following */
	0x00, 16, 32, 0, 0x01, 0x00, 0x00, 0x00,	/* GPS enable, all channels */
	0x03, 0, 0, 0, 0x00, 0x00, 0x00, 0x00,		/* BeiDou disable, 0 channels */
	0x05, 0, 0, 0, 0x00, 0x00, 0x00, 0x00,		/* QZSS disable, 0 channels */
	0x06, 0, 0, 0, 0x00, 0x00, 0x00, 0x00,		/* GLONASS disable, 0 channels */
	0xeb, 0x72					/* checksum */
};

void gps_set_gps_only(void) {

	gps_transmit_string(gpsonly, sizeof(gpsonly));
	return;
}

/*
 * gps_startup_delay
 *
 * waits for the GPS to start up. this value is empirical.
 * we could also wait for the startup string
 */
void gps_startup_delay(void) {
	/* wait for the GPS to startup */
	__delay_cycles(60000);
	__delay_cycles(60000);
	__delay_cycles(60000);
	__delay_cycles(60000);
	__delay_cycles(60000);
	__delay_cycles(60000);
	__delay_cycles(60000);
	__delay_cycles(60000);
	__delay_cycles(60000);
	__delay_cycles(60000);
	__delay_cycles(60000);
	__delay_cycles(60000);
}
